/*!
 * froala_editor v2.6.1 (https://www.froala.com/wysiwyg-editor)
 * License https://froala.com/wysiwyg-editor/terms/
 * Copyright 2014-2017 Froala Labs
 */

(function (factory) {
    if (typeof define === 'function' && define.amd) {
        // AMD. Register as an anonymous module.
        define(['jquery'], factory);
    } else if (typeof module === 'object' && module.exports) {
        // Node/CommonJS
        module.exports = function (root, jQuery) {
            if (jQuery === undefined) {
                // require('jQuery') returns a factory that requires window to
                // build a jQuery instance, we normalize how we use modules
                // that require this pattern but the window provided is a noop
                // if it's defined (how jquery works)
                if (typeof window !== 'undefined') {
                    jQuery = require('jquery');
                }
                else {
                    jQuery = require('jquery')(root);
                }
            }
            return factory(jQuery);
        };
    } else {
        // Browser globals
        factory(window.jQuery);
    }
}(function ($) {
    $.extend($.FE.POPUP_TEMPLATES, {
        'variables.picker': '[_SEARCH_][_VARIABLES_]',
        'questions.picker': '[_SEARCH_][_VARIABLES_]'
    });

    $.extend($.FE.DEFAULTS, {
        pipingSurvey: false,
        pipingSystem: false,
        pipingDistribute: false,
        pipingDistributeSms: false,
        embedQuestion: false
    });

    $.FE.PLUGINS.a4piping = function (editor) {
        var popups = {
            "variables.picker": {
                showCommand: "piping",
                insertCommand: "insertVariable",
                items: null,
                fetchMethod: a4.getSurveyAction("GetVariablesForPipingEditor", "Shared"),
                fetchParams: { includeSurvey: editor.opts.pipingSurvey, includeSystem: editor.opts.pipingSystem, includeDistribution: editor.opts.pipingDistribute, smsDistribution: editor.opts.pipingDistributeSms }
            },
            "questions.picker": {
                showCommand: "embed",
                insertCommand: "embedQuestion",
                items: null,
                fetchMethod: a4.getSurveyAction("GetQuestionsForPipingEditor", "Shared"),
                fetchParams: {}
            }
        };

        function _showPopup(popupName) {
            editor.selection.save();

            var popupDef = popups[popupName];
            var $btn = editor.$tb.find('.fr-command[data-cmd="' + popupDef.showCommand + '"]');

            var $popup = editor.popups.get(popupName);
            if (!$popup) $popup = _initPopup(popupName);

            _renderItems($popup, popupDef);

            _bindEvents($popup, popupDef);

            if (!$popup.hasClass('fr-active')) {
                // Variables popup
                editor.popups.setContainer(popupName, editor.$tb);

                // Variables popup left and top position.
                if ($btn.is(':visible')) {
                    var left = $btn.offset().left + $btn.outerWidth() / 2;
                    var top = $btn.offset().top + (editor.opts.toolbarBottom ? 10 : $btn.outerHeight() - 10);
                    editor.popups.show(popupName, left, top, $btn.outerHeight());
                } else {
                    editor.position.forSelection($popup);
                    editor.popups.show(popupName);
                }
            }
        }

        function _hidePopup(name) {
            editor.popups.hide(name);
        }

        function _initPopup(name) {
            var template = {
                search: '<div class="fr-search-variables"><input type="text" placeholder="' + richTextEditorResources.SearchVariables + '"/>' + voxco.icons.getIcon("search", "h-searchButton") + '</div>',
                variables: '<ul class="fr-variables piping-dropdown-menu"><li class="message">' + voxco.icons.getIcon("loading") + '</li></ul>'
            };

            return editor.popups.create(name, template);
        }

        function _bindEvents($popup, popupDef) {
            $popup.on("keypress", ".fr-search-variables input", function (e) {
                var code = (e.keyCode ? e.keyCode : e.which);
                if (code == 13) {
                    _searchItems($popup, popupDef, $(this).val());
                    e.preventDefault();
                }
            });

            $popup.on("click", ".h-searchButton", function () {
                _searchItems($popup, popupDef, $(".fr-search-variables input", $popup).val());
            });

            var lastLi;

            $popup.on("mouseenter", ".h-subMenuToggle", function () {
                if (lastLi)
                    $("ul", lastLi).hide();

                $("ul", $(this)).show();

                $("ul", $(this)).position({
                    my: "left top",
                    at: "right top",
                    of: $(this),
                    collision: "flipfit"
                });

                lastLi = $(this);
            });
        }

        function _searchItems($popup, popupDef, term) {
            if (term) {
                var matcher = new RegExp($.ui.autocomplete.escapeRegex(term), "i");
                var items = _.filter(_.flatten(_.map(popupDef.items, function (b) { return b.Items })), function (v) { return matcher.test(v.Label); });
                _renderItemsMenu($popup, popupDef.insertCommand, items, true);
            }
            else {
                _renderItemsMenu($popup, popupDef.insertCommand, popupDef.items);
            }
        }

        function _renderItemsMenu(container, command, items, search) {
            $(".fr-variables", container).empty();

            if (search) {
                items = items.map(function (v) {
                    return $("<li />").append($("<a />", { "class": "fr-command", "tabindex": "-1", "data-cmd": command, "data-param1": v.Value, "title": v.Label }).append(v.Label));
                });
            }
            else {
                items = items.map(function (b) {
                    var group = $("<li />", { "class": "h-subMenuToggle" }).append($("<a />").append($("<span />", { "class": "item-label" }).append(b.Group)).append(voxco.icons.getIcon("caret-right", "submenu-icon")));

                    var subItems = b.Items.map(function (v) {
                        return $("<li />").append($("<a />", { "class": "fr-command", "tabindex": "-1", "data-cmd": command, "data-param1": v.Value, "title": v.Label }).append(v.Label));
                    });

                    group.append($("<ul />", { "class": "piping-dropdown-submenu" }).append(subItems));

                    return group;
                });

                $(".fr-search-variables input", container).val("");
            }

            if (items.length > 0)
                $(".fr-variables", container).append(items);
            else
                $(".fr-variables", container).append($("<li />", { "class": "message" }).append(richTextEditorResources.NoItemsFound));
        }

        function _refresh() {
            popups["variables.picker"].items = null;
            popups["questions.picker"].items = null;
        }

        function _renderItems($popup, popupDef) {
            if (popupDef.items) {
                _renderItemsMenu($popup, popupDef.insertCommand, popupDef.items);
            }
            else {
                a4.callServerMethod(popupDef.fetchMethod, popupDef.fetchParams,
                    function (result) {
                        if (result) {
                            popupDef.items = result;
                            _renderItemsMenu($popup, popupDef.insertCommand, popupDef.items);
                        }
                    });
            }
        }

        function _insertVariable(val) {
            var sms = editor.opts.pipingDistributeSms;

            if (editor.codeView.isActive()) {
                editor.html.set($("textarea", editor.$wp).val());
            }

            _hidePopup("variables.picker");

            if (val == "$UNSUBSCRIBEURL") {
                editor.toolbar.hide();

                modalDialog.showWindow(a4.getSurveyAction("UnsubscribeLinkEditor", "Distribute", { sms: sms }), 500, 500, {
                    beforeClose: function () {
                        editor.toolbar.show();

                        var iframe = $(".fancybox-iframe").contents();
                        var text = iframe.find(".v-linkText").val();

                        var params = iframe.find(".v-customParam").map(function () {
                            if ($(this).val())
                                return $(this).attr("data-name") + "=" + $(this).val();
                            else
                                return null;
                        }).get();

                        var customPageUrl = iframe.find(".v-customPageUrl").val();

                        if (customPageUrl)
                            params.push("curl=" + customPageUrl);

                        var urlParams = (params.length > 0 ? "&" + params.join("&") : "");

                        var url = "[$UNSUBSCRIBEURL]" + urlParams;

                        editor.selection.restore();

                        if (sms) {
                            if (editor.html.get().length == 0) {
                                editor.html.set(url);
                            }
                            else {
                                editor.selection.restore();
                                editor.html.insert(url, true);
                            }
                        }
                        else
                            editor.link.insert(url, text, { "target": "_blank" });

                        if (editor.codeView.isActive()) {
                            $("textarea", editor.$wp).val(editor.html.get());
                        }
                    }
                });
            }
            else {
                if (editor.html.get().length == 0) {
                    editor.html.set("[" + val + "]");
                }
                else {
                    editor.selection.restore();
                    editor.html.insert("[" + val + "]");
                }
            }

            if (editor.codeView.isActive()) {
                $("textarea", editor.$wp).val(editor.html.get());
            }
        }

        function _embedQuestion(val) {
            var sms = editor.opts.pipingDistributeSms;

            if (editor.codeView.isActive()) {
                editor.html.set($("textarea", editor.$wp).val());
            }

            _hidePopup("questions.picker");

            if (editor.html.get().length == 0) {
                editor.html.set("[@" + val + "]");
            }
            else {
                editor.selection.restore();
                editor.html.insert("[@" + val + "]");
            }

            if (editor.codeView.isActive()) {
                $("textarea", editor.$wp).val(editor.html.get());
            }
        }

        return {
            showPopup: _showPopup,
            hidePopup: _hidePopup,
            insertVariable: _insertVariable,
            embedQuestion: _embedQuestion,
            refresh: _refresh
        };
    };

    $.FE.DefineIcon('piping', {
        NAME: 'magic'
    });

    $.FE.RegisterCommand('piping', {
        title: richTextEditorResources.InsertVariable,
        undo: false,
        focus: true,
        refreshOnCallback: false,
        popup: true,
        callback: function () {
            var popupName = 'variables.picker';
            this.events.disableBlur();

            if (!this.popups.isVisible(popupName)) {
                this.a4piping.showPopup(popupName);
            } else {
                if (this.$el.find('.fr-marker').length) {
                    this.selection.restore();
                }
                this.a4piping.hidePopup(popupName);
            }
        },
        plugin: 'a4piping'
    });

    $.FE.RegisterCommand('insertVariable', {
        undo: false,
        focus: true,
        callback: function (cmd, val) {
            this.a4piping.insertVariable(val);
            this.events.enableBlur();
        }
    });

    $.FE.DefineIcon('embed', {
        NAME: 'file-code-o'
    });

    $.FE.RegisterCommand('embed', {
        title: richTextEditorResources.EmbedQuestion,
        undo: false,
        focus: true,
        refreshOnCallback: false,
        popup: true,
        callback: function () {
            var popupName = 'questions.picker';
            this.events.disableBlur();

            if (!this.popups.isVisible(popupName)) {
                this.a4piping.showPopup(popupName);
            } else {
                if (this.$el.find('.fr-marker').length) {
                    this.selection.restore();
                }
                this.a4piping.hidePopup(popupName);
            }
        },
        plugin: 'a4piping'
    });

    $.FE.RegisterCommand('embedQuestion', {
        undo: false,
        focus: true,
        callback: function (cmd, val) {
            this.a4piping.embedQuestion(val);
            this.events.enableBlur();
        }
    });
}));
